function t_debug(mode)

load t_debug_config

% This function T_DEBUG will turn textual debug mode 'on' or 'off'. 
% This follows the philosophy described by Brian Kernighan and Rob Pike 
% in "The Practice of Programming":
% 
% "... we find stepping through a program less productive than thinking 
% harder and adding output statements and self-checking code at critical 
% places. Clicking over statements takes longer than scanning the output 
% of judiciously-placed displays. It takes less time to decide where to 
% put print statements than to single-step to the critical section of code, 
% even assuming we know where that is. More important, debugging statements 
% stay with the program; debugging sessions are transient."
%
% To allow for debugging statements that stay with the program and can be 
% switched on and off, I wrote two very simple Matlab functions 
% (t_debug.m, print_debug.m). Here is how they work:
%
% >> t_debug on
% >> print_debug('t_debug is on');
% DEBUG: t_debug is on
%
% >> t_debug
% DEBUG state: 1
%
% >> t_debug off
% >> print_debug('This should not print');
%
% >> t_debug offff
% Warning: USE: debug(mode) with mode='on', 'off', or a numeric value to
% set debugging to a certain level. You used 'offff'. 
% > In t_debug at 54
%
% >> t_debug(10)
% >> print_debug('t_debug prints up to level 10', 9)
% DEBUG: (L=9)	t_debug prints up to level 10
% >> print_debug('t_debug prints up to level 10 (this won''t print)', 11)
% >> print_debug('t_debug prints up to level 10')
% DEBUG: t_debug prints up to level 10
%
% >> t_debug
% DEBUG state: 10
%
% Author: Ronni Grapenthin, UAF-GI, AVO
% See also: PRINT_DEBUG
% Last Modified: 2010-11-22

    %print current state, if no argument is given
    if(nargin == 0)
        fprintf('DEBUG state: %d\n', getpref(T_DBG_GROUP, T_DBG_PREF_DEBUG));
        return
    end

    % the way this works is by opening a new preference category and
    % setting the value for debug to either true or false. This could be
    % extended by various levels of 'chattiness'
    if( strcmp(mode, T_DBG_ON) )
        setpref(T_DBG_GROUP, T_DBG_PREF_DEBUG, true);
    elseif( strcmp(mode, T_DBG_OFF) )
        setpref(T_DBG_GROUP, T_DBG_PREF_DEBUG, false);
    elseif(isnumeric(mode))
        setpref(T_DBG_GROUP, T_DBG_PREF_DEBUG, mode);   %set to a certain level
    else
        warning('MATLAB:print_debug:wrongArgument', ...
            'USE: t_debug(mode) with mode=''%s'', ''%s'', or a numeric value to set debugging to a certain level. You used ''%s''.', T_DBG_ON, T_DBG_OFF, mode);
    end
return
